//   read92
//
// This program downloads the memory of a PRO-92, and writes a SPG file.
//
// Written by Ken Plotkin
//            kjp15@cornell.edu
//            December 2000
//            February 2000
//
// Translated from Fortran to C by Steve Falco
// 				   sfalco@worldnet.att.net
// 				   September 2001
// 				   This version is "Unix-centric" in that it
// 				   depends on various Unix system calls to
// 				   establish the parameters of the serial
// 				   connection.  Debugged and tested on RedHat
// 				   Linux version 7.1.
//
// Distributed as careware.  If you like it, donate some money to a worthwhile
// charity.

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/poll.h>

typedef unsigned char UCHAR;
typedef unsigned int UINT;

#define BLEN	26496	// Body length
#define MBLEN	30000	// Body length (plus some margin)
#define TLEN	6272	// Trailer length
#define EOL	"\n"	// end of line
#define TRYS	1000	// how many positions we scan looking for start of data

char	*port = "/dev/ttyS0";
char	*outfile = "dump.spg";

struct termios new_settings;
struct pollfd peek[1];

void use();

main(int argc, char *argv[])
{
	extern char *optarg;
	extern int optind, opterr, optopt;

	UCHAR	clndat[MBLEN];
	UCHAR	revdat[MBLEN];
	UCHAR	trail[TLEN];
	UCHAR	errbuf[BUFSIZ];

	UCHAR	dump[]	= { 0xcd };
	UCHAR	fin[]	= { 0x02, 0x00, 0x00, 0x00, 0x00 };
	UCHAR	cstart[]= { 0x01, 0x98, 0x03, 0x52 };

	UCHAR	*dot;
	UCHAR	flag;

	int	infd, outfd, ic, rv, nchars, i, idat;

	time_t  starttime, delta;

	starttime = time(0);

	while((ic = getopt(argc, argv, "hp:")) != -1) {
		switch(ic) {
			case 'p':
				port = optarg;
				break;
			case 'h': /*FALLTHROUGH*/
			case '?':
				use();
				break;
		}
	}


	if(optind < argc) {
		outfile = argv[optind];
	}

	// Keep the user out of trouble - part 1
	if( !(					     // negate the whole thing
		((dot = strrchr(outfile, '.')) != 0) // we found the last dot
		&&				     // and
		( 				     // (
		    (strcmp(dot, ".spg") == 0)	     // we have .spg
		    ||				     // or
		    (strcmp(dot, ".SPG") == 0)	     // we have .spg
		)				     // )
	)) {
		fprintf(stderr, "File name (%s) must end with .spg" EOL, outfile);
		exit(1);
	}

	// Open the SPG output file
	if((outfd = open(outfile, O_WRONLY | O_CREAT, 0644)) == -1) {
		sprintf(errbuf, "Cannot open %s", outfile);
		perror(errbuf);
		exit(1);
	}

	// Start the buffers off as empty.
	memset(clndat, 0, MBLEN);
	memset(trail, 0, TLEN);

	// Open serial port to scanner.  Parameters are:
	// 	4800 baud
	// 	8 data bits
	// 	2 stop bits
	// 	even parity
	if((infd = open(port, O_RDWR)) == -1) {
		sprintf(errbuf, "Cannot open %s", port);
		perror(errbuf);
		exit(1);
	}
	new_settings.c_iflag = 0;
	new_settings.c_oflag = 0;
	new_settings.c_cflag = 0;
	new_settings.c_lflag = 0;
	if(cfsetospeed(&new_settings, B4800) == -1) {
		sprintf(errbuf, "Cannot set baud rate for %s", port);
		perror(errbuf);
		exit(1);
	}
	new_settings.c_cflag |= CREAD;		// enable input
	new_settings.c_cflag |= CLOCAL;		// ignore all control lines
	new_settings.c_cflag |= CS8;		// want 8 data bits
	new_settings.c_cflag |= CSTOPB;		// want 2 stop bits
	new_settings.c_cflag |= PARENB;		// parity is enabled
	new_settings.c_cflag &= ~PARODD;	// and even
	if(tcsetattr(infd, TCSANOW, &new_settings) == -1) {
		sprintf(errbuf, "Cannot set parameters for %s", port);
		perror(errbuf);
		exit(1);
	}

	// Set DTR (+12 volts), clear RTS (-12 volts).  These pins power
	// the hardware interface.  For some reason, DTR has to be set before
	// RTS in order for this to work correctly.
	flag = TIOCM_DTR;
	if(ioctl(infd, TIOCMBIS, &flag)  == -1) {
		sprintf(errbuf, "Cannot set DTR for %s", port);
		perror(errbuf);
		exit(1);
	}
	flag = TIOCM_RTS;
	if(ioctl(infd, TIOCMBIC, &flag)  == -1) {
		sprintf(errbuf, "Cannot clear RTS for %s", port);
		perror(errbuf);
		exit(1);
	}

	// Read data
	nchars = 0;

	// Send dump command
	if(write(infd, dump, 1) != 1) {
		sprintf(errbuf, "Cannot write dump command to %s", port);
		perror(errbuf);
		exit(1);
	}
	if(tcdrain(infd) == -1) {
		sprintf(errbuf, "Cannot drain output to %s", port);
		perror(errbuf);
		exit(1);
	}

	// Wait up to 200 ms for data to be available.  If available, transfer
	// to clndat.  If timed out, then we are done.
	while(1) {
		peek[0].fd = infd;
		peek[0].events = POLLIN;
		rv = poll(peek, 1, 200);
		if(rv == 0) {
			break; // timed out
		}
		if(rv != 1) {
			sprintf(errbuf, "poll() failed while waiting for input");
			perror(errbuf);
			exit(1);
		}
		if((rv = read(infd, &clndat[nchars + 1], MBLEN - (nchars + 1))) == -1) {
			sprintf(errbuf, "read() failed");
			perror(errbuf);
			exit(1);
		}
		nchars += rv;

		// Print the percentage we have completed.  Note the '\r' so we
		// overwrite previous values.
		fprintf(stderr, "%3d%%\r", nchars/265);
	}
	fprintf(stderr, EOL);

	// There will be some filler before the real data, so look for the
	// start of data string (0x01 0x98 0x03 0x52).
	for(i = 4; i <= TRYS; i++) {
		if(!strncmp(&clndat[i - 3], cstart, 4)) {
			idat = i; // Found it
			break;
		}
	}
	if(i > TRYS) {
		// We never found the data
		fprintf(stderr, "Cannot find start sequence" EOL);
		exit(1);
	}

#if DEBUG
	fprintf(stderr, EOL "data start at %d, end at %d, end marker is 0x%02x" EOL,
		idat + 1, idat + BLEN, clndat[idat + BLEN + 1]);
#endif

	// Sadly, the data come in backwards, so we have to reverse the array
	for(i = 0; i < BLEN; i++) {
		revdat[i] = clndat[idat + BLEN - i];
	}

	// Now write everything to the .spg file
	if(write(outfd, fin, 5) != 5) {
		sprintf(errbuf, "Cannot write header to %s", outfile);
		perror(errbuf);
		exit(1);
	}
	if(write(outfd, revdat, BLEN) != BLEN) {
		sprintf(errbuf, "Cannot write spg data to %s", outfile);
		perror(errbuf);
		exit(1);
	}
	if(write(outfd, trail, TLEN) != TLEN) {
		sprintf(errbuf, "Cannot write trailer to %s", outfile);
		perror(errbuf);
		exit(1);
	}
	delta = (int)(time(0) - starttime);

	if(nchars == (BLEN + idat + 1)) {
		fprintf(stderr, "Done!  Enjoy your data file.  (time was %02ld:%02ld)" EOL, delta / 60, delta % 60);
		
	} else {
		fprintf(stderr, "Error - received %d bytes instead of expected %d" EOL, nchars, (BLEN + idat + 1));
		exit(1);
	}

	close(outfd);
	close(infd);

	exit(0);
}

void
use()
{
	fprintf(stderr, "read92 [-h] [-p port] [filename.spg]" EOL);
	fprintf(stderr, "  -h prints this help message" EOL);
	fprintf(stderr, "  -p port sets the device where the scanner is connected" EOL);
	fprintf(stderr, "          (typically /dev/ttyS0 or /dev/ttyS1," EOL);
	fprintf(stderr, "          defaults to %s)" EOL, port);
	fprintf(stderr, "  filename.spg is the output file which must end with .spg" EOL);
	fprintf(stderr, "          (defaults to %s)" EOL, outfile);

	exit(1);
}
